/**
  ******************************************************************************
  * @file           : dfu_update.c
  * @brief          : Simple example for showing dfu update functionality
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Kuenzler Technologies GbR
  * All rights reserved.
  *
  *
  ******************************************************************************
*/

/*==============================================================================*/
/* Includes                                                                  	*/
/*==============================================================================*/
#include "bootloader_update.h"
#include <string.h>
#include "stdlib.h"

/*==============================================================================*/
/* Defines & Macros                                                           	*/
/*==============================================================================*/

/*==============================================================================*/
/* Public variables                                                           	*/
/*==============================================================================*/

/*==============================================================================*/
/* Private variables                                                          	*/
/*==============================================================================*/

/* Configuration written in flash - 配置写入闪存*/
static dfu_config_t *cfgFlash   = (dfu_config_t*) (FLASH_ADDR_CONFIG);

/*==============================================================================*/
/* Private functions                                                          	*/
/*==============================================================================*/
static bool dfu_clear_request(void);
static bool write_updated_bootloader(void);
/*==============================================================================*/
/* Function definitions                                                     	*/
/*==============================================================================*/
void UpdateBootloader()
{
	write_updated_bootloader();
	dfu_clear_request();
}

/*==============================================================================*/
/* Private Functions                                                     	*/
/*==============================================================================*/
static bool dfu_clear_request()
{
	// Calculate aligned length for compatible flash layout
	uint16_t alignedLength = sizeof(dfu_config_t) / FLASH_BYTE_ALIGNMENT;
	if(sizeof(dfu_config_t) % FLASH_BYTE_ALIGNMENT > 0)
	{
		alignedLength += FLASH_BYTE_ALIGNMENT;
	}
	// Reserve buffer for any alignemnt
	uint8_t* writeBuffer = malloc(alignedLength * FLASH_BYTE_ALIGNMENT);
	if(writeBuffer == NULL)
	{
		return false;
	}
	memcpy(writeBuffer,cfgFlash, sizeof(dfu_config_t));
	/* Set Signature */
	dfu_config_t* config = (dfu_config_t*)writeBuffer;
	config->reboot_cause = BOOT_FIRST_TIME;
	config->bootloaderVersion[0] = BOOTLOADER_UPDATED_VERSION_MAJOR;
	config->bootloaderVersion[1] = BOOTLOADER_UPDATED_VERSION_MINOR;
	config->bootloaderVersion[2] = BOOTLOADER_UPDATED_VERSION_PATCH;
	config->bootloaderVersion[3] = BOOTLOADER_UPDATED_VERSION_RELEASE;
	/* Pages needed for saving the settings - 保存设置所需的页面*/
	if(!HardwareEraseFlash(FLASH_ADDR_CONFIG, FLASH_PAGE_CONFIG, 1))
	{
		free(writeBuffer);
		return false;
	}
	if(!HardwareWriteFlash(FLASH_ADDR_CONFIG, alignedLength, writeBuffer))
	{
		free(writeBuffer);
		return false;
	}
	free(writeBuffer);
	return true;
}


static bool write_updated_bootloader()
{
	/* Pages needed for Application - 申请所需页面*/
	uint8_t pages = (uint8_t)(sizeof(bootloaderBinary) / FLASHPAGE_SIZE);
	if((sizeof(bootloaderBinary) % FLASHPAGE_SIZE) > 0)
	{
		pages++;
	}
	 /* Pages needed for Bootloader */
	if(!HardwareEraseFlash(FLASH_ADDR_BOOTLOADER, FLASH_PAGE_BOOTLOADER, pages))
	{
		return false;
	}


	uint32_t offset = 0;
	uint32_t length = sizeof(bootloaderBinary);
	uint8_t alignedBuffer[FLASH_PACKET_SIZE];

	while (offset < length)
	{
		uint32_t remaining = length - offset;
		uint32_t chunkSize = remaining > FLASH_PACKET_SIZE ? FLASH_PACKET_SIZE : remaining;

		// Align chunkSize up to next multiple of FLASH_ALIGNMENT_BYTES
		uint32_t alignedChunkSize = (chunkSize + FLASH_BYTE_ALIGNMENT - 1) & ~(FLASH_BYTE_ALIGNMENT - 1);

		// Clear buffer to avoid writing garbage on unaligned tail
		memset(alignedBuffer, 0xFF, alignedChunkSize); // 0xFF is typical erased flash value

		// Copy actual data
		memcpy(alignedBuffer, bootloaderBinary + offset, chunkSize);

		if (!HardwareWriteFlash(FLASH_ADDR_BOOTLOADER + offset, alignedChunkSize / FLASH_BYTE_ALIGNMENT, alignedBuffer))
		{
			return false;
		}

		offset += chunkSize;
	}
	return true;
}

